'use strict';

export class TagsField {
	constructor(disallowed_tag_prefixes, placeholder, remove_tag_title, initial_tags) {
		this.tags_list = [];
		this.disallowed_tag_prefixes = disallowed_tag_prefixes;
		if ('undefined' == typeof placeholder) {
			placeholder = 'Enter a tag to add, click a tag to remove';
		}
		if ('undefined' == typeof remove_tag_title) {
			remove_tag_title = 'Remove this tag';
		}
		this.placeholder = placeholder;
		this.remove_tag_title = remove_tag_title;
		// create field input
		this.input = document.createElement('input');
		this.input.type = 'text';
		this.input.placeholder = this.placeholder;
		// listener for input finish
		this.debounce = null;
		this.input.addEventListener('keydown', e => {
			// ignore modifier keys
			if (-1 != ['Meta', 'Shift', 'Control', 'Alt'].indexOf(e.key)) {
				return;
			}
			// not finishing input
			if ('Enter' != e.key) {
				// navigation up in suggestion list
				if ('ArrowUp' == e.key) {
					this.current_tag_suggestion_index -= 1;
					this.update_tag_suggestion_highlight();
					return;
				}
				// navigation down in suggestion list
				else if ('ArrowDown' == e.key) {
					this.current_tag_suggestion_index += 1;
					this.update_tag_suggestion_highlight();
					return;
				}
				// normal key
				if (this.debounce) {
					clearTimeout(this.debounce);
				}
				this.hide_suggestions();
				this.debounce = setTimeout(() => {
					this.show_suggestions();
				}, 250);
				return;
			}
			// suggestion is highlighted
			if (this.current_tag_suggestion_index > -1) {
				e.preventDefault();
				let current_tag_suggestion = this.tag_suggestions_list.children[this.current_tag_suggestion_index];
				if (current_tag_suggestion) {
					this.add_tag(current_tag_suggestion.value);
					this.current_tag_suggestion_index = -1;
				}
				return;
			}
			if ('' == this.input.value) {
				// dispatch submit
				this.input.dispatchEvent(
					new CustomEvent('submit')
				);
				return;
			}
			// prevent form submit in case we're inside of a form
			e.preventDefault();
			e.stopPropagation();
			this.add_tags(this.to_list(this.input.value));
			setTimeout(() => {
				this.clear_input();
				this.hide_suggestions();
			}, 1);
		});
		this.input.addEventListener('focusout', e => {
			this.hide_suggestions();
		});
		this.input.addEventListener('focus', e => {
			this.show_suggestions();
		});
		this.preview = document.createElement('div');
		// suggestions lists
		this.current_tag_suggestion_index = -1;
		this.tag_suggestions_limit = 16;
		// combined tag suggestions should be stored in a meta element for all tag fields to access
		this.tag_suggestions_combined = document.querySelector('#tag-suggestions-combined');
		if (!this.tag_suggestions_combined) {
			this.tag_suggestions_combined = document.createElement('meta');
			this.tag_suggestions_combined.id = 'tag-suggestions-combined';
			this.tag_suggestions_combined.suggestions = [];
			document.head.appendChild(this.tag_suggestions_combined);
		}
		// actual tag suggestions element should be stored in a div element for all tag fields to access
		this.tag_suggestions_list = document.querySelector('#tag-suggestions-list');
		if (!this.tag_suggestions_list) {
			this.tag_suggestions_list = document.createElement('div');
			this.tag_suggestions_list.id = 'tag-suggestions-list';
			this.hide_suggestions();
			document.body.appendChild(this.tag_suggestions_list);
		}

		// add reference to this tag editor on its preview and input
		this.input.tag_editor = this;
		this.preview.tag_editor = this;
		// add initial tags
		if ('undefined' != typeof initial_tags) {
			this.clear();
			if ('string' == typeof initial_tags) {
				initial_tags = this.to_list(initial_tags);
			}
			this.add_tags(initial_tags);
		}

		// allow or disallow meta tag suggestions
		this.meta_tag_suggestions = true;
	}
	update_tag_suggestion_highlight() {
		// de-highlist all
		let tag_suggestions = this.tag_suggestions_list.children;
		for (let i = 0; i < tag_suggestions.length; i++) {
			tag_suggestions[i].classList.remove('highlighted');
		}
		if (this.current_tag_suggestion_index < 0) {
			this.current_tag_suggestion_index = -1;
			return;
		}
		if (this.current_tag_suggestion_index > this.tag_suggestions_list.children.length - 1) {
			this.current_tag_suggestion_index = this.tag_suggestions_list.children.length - 1;
		}
		let current_tag_suggestion = this.tag_suggestions_list.children[this.current_tag_suggestion_index];
		if (current_tag_suggestion) {
			current_tag_suggestion.classList.add('highlighted');
		}
	}
	show_suggestions() {
		this.hide_suggestions();
		if (
			'' == this.input.value
			|| 1 > this.tag_suggestions_limit
		) {
			return;
		}
		let needle = this.input.value.trim().replace(/\\/g, '\\\\');
		let negation = false;
		if ('-' == needle[0]) {
			negation = true;
			needle = needle.substring(1);
		}

		let meta_tag_prefixes = [
			'orientation:',
			'order by:',
			'sort:',
			'category:',
			'mimetype:',
		];

		let r = new RegExp(needle, 'i');
		let dupe_check = [];
		let suggestions = this.tag_suggestions_combined.suggestions;
		for (let i = 0; i < suggestions.length; i++) {
			let suggestion = suggestions[i];
			// not including meta tag suggestions
			if (!this.meta_tag_suggestions) {
				let skip = false;
				for (let i = 0; i < meta_tag_prefixes.length; i++) {
					let meta_tag_prefix = meta_tag_prefixes[i];
					// check if current suggestion starts with meta tag prefix
					if (suggestion.substring(0, meta_tag_prefix.length) == meta_tag_prefix) {
						skip = true;
						break;
					}
				}
				if (skip) {
					continue;
				}
			}
			// found a suggestion
			if (
				r.test(suggestion)
				&& -1 == dupe_check.indexOf(suggestion)
			) {
				// 
				dupe_check.push(suggestion);
				if (negation) {
					suggestion = '-' + suggestion;
				}
				// add suggestion to visible suggestions list
				let suggestion_el = document.createElement('div');
				suggestion_el.value = suggestion;
				suggestion_el.innerText = suggestion;
				suggestion_el.classList.add('suggestion');
				suggestion_el.addEventListener('mousedown', e => {
					this.add_tag(e.currentTarget.value);
				});
				this.tag_suggestions_list.appendChild(suggestion_el);
				// stop looking for suggestions if at the tag suggestions limit
				if (this.tag_suggestions_list.children.length == this.tag_suggestions_limit) {
					break;
				}
			}
		}
		this.tag_suggestions_list.classList.add('visible');
	}
	hide_suggestions() {
		this.tag_suggestions_list.innerHTML = '';
		this.tag_suggestions_list.classList.remove('visible');
	}
	clear() {
		//what
		this.input.dispatchEvent(
			new CustomEvent('fetched')
		);
		this.clear_tags();
		this.clear_preview();
		this.clear_input();
	}
	clear_tags() {
		this.tags_list = [];
	}
	clear_preview() {
		this.preview.innerHTML = '';
	}
	clear_input() {
		this.input.value = '';
		this.hide_suggestions();
	}
	discard() {
		this.clear();
	}
	to_list(tags_string) {
		if ('' == tags_string) {
			return [];
		}
		// single tag
		if (-1 == tags_string.indexOf('#')) {
			return [tags_string.trim()];
		}
		// strip leading and trailing hashes and split on hashes
		let tags_list_raw = tags_string.replace(/^#+|#+$/g, '').split('#');
		let tags_list = [];
		for (let i = 0; i < tags_list_raw.length; i++) {
			tags_list.push(tags_list_raw[i].trim());
		}
		return tags_list;
	}
	//TODO needs to separate out tag prefix for certain tags so that they can be styled appropriately
	create_tag_element(tag, title, link_uri) {
		tag = tag.trim();
		let el = document.createElement('span');
		el.classList.add('tag');
		el.dataset.tag = tag.replace(/"/g, '&quot;');
		if ('undefined' != typeof title) {
			el.title = title;
		}
		let inner_el = null;
		if ('undefined' == typeof link_uri) {
			inner_el = document.createElement('span');
		}
		else {
			inner_el = document.createElement('a');
			inner_el.href = link_uri.replace('{}', encodeURIComponent(tag));
		}
		// for negation tag shouldn't include the hyphen in the actual text
		inner_el.innerText = '#' + ('-' == tag[0] ? tag.substring(1) : tag);
		el.appendChild(inner_el);
		return el;
	}
	is_disallowed(tag) {
		for (let i = 0; i < this.disallowed_tag_prefixes.length; i++) {
			let tag_prefix = this.disallowed_tag_prefixes[i];
			if (tag_prefix == tag.substring(0, tag_prefix.length)) {
				return true;
			}
		}
		return false;
	}
	add_tag(tag) {
		this.add_tags([tag]);
	}
	add_tags(tags_list) {
		this.clear_input();
		for (let i = 0; i < tags_list.length; i++) {
			let tag = tags_list[i].trim();
			if (
				-1 != this.tags_list.indexOf(tag)
				|| this.is_disallowed(tag)
			) {
				continue;
			}
			this.tags_list.push(tag);
			let el = this.create_tag_element(tag)
			el.title = this.remove_tag_title;
			el.addEventListener('click', e => {
				this.remove_tag(e.currentTarget.dataset.tag);
			});
			this.preview.appendChild(el);
			this.hide_suggestions();
			// dispatch add event
			this.input.dispatchEvent(
				new CustomEvent(
					'added',
					{
						detail:
						{
							tag: tag,
							el: el,
						}
					}
				)
			);
		}
	}
	remove_tag(tag) {
		this.remove_tags([tag]);
	}
	remove_tags(tags_list) {
		for (let i = 0; i < tags_list.length; i++) {
			let tag = tags_list[i].replace(/&quot;/g, '"').trim();
			let tag_index = this.tags_list.indexOf(tag)
			if (-1 == tag_index) {
				continue;
			}
			this.tags_list.splice(tag_index, 1);
			let tag_el = this.preview.querySelector('.tag[data-tag="' + tag.replace(/"/g, '&quot;').replace(/\\/g, '\\\\') + '"]');
			tag_el.parentNode.removeChild(tag_el);
			// dispatch remove event
			this.input.dispatchEvent(
				new CustomEvent(
					'removed',
					{
						detail:
						{
							tag: tag,
						}
					}
				)
			);
		}
		this.input.focus();
	}
	to_string(tags_list) {
		tags_list.sort();
		return tags_list.join('#');
	}
	fetch_suggestions() {
		fetch_tag_suggestions();
		let tag_suggestion_lists = document.querySelectorAll('meta[name="tag-suggestion-list"]');
		for (let i = 0; i < tag_suggestion_lists.length; i++) {
			let meta_tag = tag_suggestion_lists[i];
			if (!meta_tag.hasOwnProperty('fetched')) {
				continue;
			}
			// this list hasn't been fetched yet, so wait for it and add suggestions to combined list when it's done
			if (!meta_tag.fetched) {
				meta_tag.addEventListener('fetched', e => {
					for (let j = 0; j < e.currentTarget.suggestions.length; j++) {
						let suggestion = e.currentTarget.suggestions[j];
						this.tag_suggestions_combined.suggestions.push(suggestion);
					}
				});
			}
			else {
				for (let j = 0; j < meta_tag.suggestions.length; j++) {
					let suggestion = meta_tag.suggestions[j];
					this.tag_suggestions_combined.suggestions.push(suggestion);
				}
			}
		}
	}
}

// fetches each meta tag-suggestion-list and adds the response to the meta tag as an array of tags
export function fetch_tag_suggestions() {
	let tag_suggestion_lists = document.querySelectorAll('meta[name="tag-suggestion-list"]');
	for (let i = 0; i < tag_suggestion_lists.length; i++) {
		let meta_tag = tag_suggestion_lists[i];
		//what
		if (meta_tag.hasOwnProperty('suggestions')) {
			continue;
		}
		meta_tag.suggestions = [];
		meta_tag.fetched = false;
		let xhr = new XMLHttpRequest();
		xhr.meta_tag = meta_tag;
		xhr.onreadystatechange = () => {
			if (xhr.readyState == XMLHttpRequest.DONE) {
				if (200 == xhr.status) {
					if (xhr.response) {
						for (let i = 0; i < xhr.response.length; i++) {
							xhr.meta_tag.suggestions.push(xhr.response[i])
						}
					}
				}
				xhr.meta_tag.fetched = true;
				xhr.meta_tag.dispatchEvent(
					new CustomEvent('fetched')
				);
			}
		};
		let action = tag_suggestion_lists[i].getAttribute('content');
		xhr.open('GET', action + (-1 != action.indexOf('?') ? '&' : '?') + '_' + new Date().getTime(), true);
		xhr.responseType = 'json';
		if ('1' == meta_tag.dataset.credentials) {
			xhr.withCredentials = true;
		}
		xhr.send();
	}
}
