'use strict';

export class SimpleComments {
	constructor(config) {
		this.config = config;
		if (!this.config['list_mode']) {
			this.config['list_mode'] = false;
		}
		if (!this.config['target_selector']) {
			this.config['target_selector'] = '[data-id]';
		}
		if (!this.config['comments_reference_selector']) {
			this.config['comments_reference_selector'] = null;
		}
		if (!this.config['comments_insert_mode']) {
			this.config['comments_insert_mode'] = 'append';
		}
		if (!this.config['comments_order']) {
			this.config['comments_order'] = 'newest_first';
		}

		this.base_uri = document.querySelector('base').href;
		this.comments_uri = this.base_uri + 'simple_comments/';

		this.authorized = false;

		this.comments = null;
		this.comments_table = null;
		this.target_id = null;
		this.form = null;

		// form and comments
		if (!this.config['list_mode']) {
			let target_el = document.querySelector(this.config['target_selector']);
			if (!target_el) {
				console.log('No comment target found');
				return;
			}
			this.target_id = target_el.dataset.id;

			this.create_form();
			this.check_authorized();

			let comments_reference_el = document.querySelector(this.config['comments_reference_selector']);
			if (!comments_reference_el) {
				return;
			}

			this.comments = document.createElement('div');
			this.comments.id = 'simple-comments';

			this.comments_table = document.createElement('table');
			this.comments_table.id = 'comments-table';

			if ('append' == this.config['comments_insert_mode']) {
				comments_reference_el.appendChild(this.comments);
			}
			else if ('before' == this.config['comments_insert_mode']) {
				comments_reference_el.parentNode.insertBefore(this.comments, comments_reference_el);
			}
			else if ('after' == this.config['comments_insert_mode']) {
				comments_reference_el.parentNode.insertBefore(this.comments, comments_reference_el.nextSibling);
			}

			if ('newest_first' == this.config['comments_order']) {
				this.comments.appendChild(this.form);
				this.comments.appendChild(this.comments_table);
			}
			else {
				this.comments.appendChild(this.comments_table);
				this.comments.appendChild(this.form);
			}

			this.display_comments(this.target_id);
		}
		else {
			//TODO go through all targets and fetch their comments just to get comment counts and add comment count indicators to them
		}
	}
	create_form() {
		// form
		this.form = document.createElement('form');
		this.form.action = this.comments_uri;
		this.form.method = 'POST';
		this.form.id = 'add-comment';
		let name_input = document.createElement('input');
		name_input.type = 'text';
		name_input.name = 'name';
		name_input.placeholder = 'Name';
		this.form.appendChild(name_input);
		let comment_input = document.createElement('input');
		comment_input.type = 'text';
		comment_input.name = 'comment';
		comment_input.placeholder = 'Comment';
		this.form.appendChild(comment_input);
		let submit_input = document.createElement('input');
		submit_input.type = 'submit';
		submit_input.value = 'Submit comment';
		this.form.appendChild(submit_input);
		if (this.authorized) {
			this.add_manage_comments_link();
		}
		this.form.addEventListener('submit', e => {
			e.preventDefault();
			// clear form errors
			let errors = this.comments.querySelectorAll('.error');
			for (let i = 0; i < errors.length; i++) {
				let error = errors[i];
				this.comments.remove(error);
			}
			// submit comment
			let xhr = new XMLHttpRequest();
			let fd = new FormData(this.form);
			fd.append('target_id', this.target_id);
			xhr.responseType = 'json';
			xhr.onreadystatechange = () => {
				if (xhr.readyState == XMLHttpRequest.DONE) {
					if (200 == xhr.status) {
						if (!xhr.response) {
							return;
						}
						if (!xhr.response['approved']) {
							let comment_row_el = document.createElement('tr');
							let comment_cell_el = document.createElement('td');
							comment_cell_el.innerText = 'Your comment was submitted and is pending approval';
							comment_cell_el.setAttribute('colspan', 3);
							comment_row_el.appendChild(comment_cell_el);
							if ('newest_first' == this.config['comments_order']) {
								this.comments_table.insertBefore(comment_row_el, this.comments_table.firstChild);
							}
							else {
								this.comments_table.append(comment_row_el);
							}

						}
						// add comment
						this.display_comment(xhr.response);
					}
					else {
						if (!xhr.response) {
							xhr.response = 'Unknown error';
						}
						// show error above form
						let error = document.createElement('div');
						error.classList.add('error');
						error.innerText = xhr.response;
						this.comments.insertBefore(error, this.form);
					}
				}
			};
			xhr.open(
				'POST',
				this.form.action + (-1 != this.form.action.indexOf('?') ? '&' : '?') + '_' + new Date().getTime(),
				true
			);
			xhr.withCredentials = true;
			xhr.send(fd);
			// clear form
			this.form.reset();
		});
	}
	fetch_comments(target_id, cb) {
		let xhr = new XMLHttpRequest();
		xhr.responseType = 'json';
		xhr.onreadystatechange = () => {
			if (xhr.readyState == XMLHttpRequest.DONE) {
				if (200 == xhr.status) {
					cb(xhr.response);
				}
			}
		};
		xhr.open(
			'GET',
			this.comments_uri + 'comments/' + target_id + '.json' + '?_' + new Date().getTime(),
			true
		);
		xhr.send('');
	}
	display_comment(comment) {
		if (!comment['approved']) {
			return;
		}
		let comment_row_el = document.createElement('tr');
		comment_row_el.classList.add('comment');
		comment_row_el.dataset.id = comment['id'];
		let time_cell_el = document.createElement('td');
		time_cell_el.classList.add('time');
		let date = new Date(comment['time'] * 1000);
		time_cell_el.innerText = date.toDateString();
		comment_row_el.appendChild(time_cell_el);
		let name_cell_el = document.createElement('td');
		name_cell_el.classList.add('name');
		name_cell_el.innerText = comment['name'];
		comment_row_el.appendChild(name_cell_el);
		let text_cell_el = document.createElement('td');
		text_cell_el.classList.add('text');
		text_cell_el.innerText = comment['text'];
		comment_row_el.appendChild(text_cell_el);
		// remove link
		if (this.authorized) {
			this.add_remove_link(comment_row_el);
		}
		if ('newest_first' == this.config['comments_order']) {
			this.comments_table.insertBefore(comment_row_el, this.comments_table.firstChild);
		}
		else {
			this.comments_table.append(comment_row_el);
		}
	}
	display_comments(target_id) {
		this.fetch_comments(target_id, (comments) => {
			for (let i = 0; i < comments.length; i++) {
				this.display_comment(comments[i]);
			}
		});
	}
	check_authorized() {
		let xhr = new XMLHttpRequest();
		let fd = new FormData();
		fd.append('check_authorized', true)
		xhr.responseType = 'json';
		xhr.onreadystatechange = () => {
			if (xhr.readyState == XMLHttpRequest.DONE) {
				if (200 == xhr.status) {
					this.authorized = true;
					this.add_remove_links();
					this.add_manage_comments_link();
				}
			}
		};
		xhr.open(
			'POST',
			this.comments_uri + '?_' + new Date().getTime(),
			true
		);
		xhr.withCredentials = true;
		xhr.send(fd);
	}
	add_manage_comments_link() {
		// already added
		if (!this.form || this.form.querySelector('a')) {
			return;
		}
		let mod_link = document.createElement('a');
		mod_link.innerText = 'Manage comments';
		mod_link.href = this.comments_uri;
		this.form.appendChild(mod_link);
	}
	add_remove_links() {
		// already added
		let first_comment_cells = document.querySelectorAll('tr.comment:first-child td');
		if (4 == first_comment_cells.length) {
			return;
		}
		let comment_row_els = document.querySelectorAll('tr.comment');
		for (let i = 0; i < comment_row_els.length; i++) {
			this.add_remove_link(comment_row_els[i]);
		}
	}
	add_remove_link(comment_row_el) {
		let comment_id = comment_row_el.dataset.id;
		let remove_cell_el = document.createElement('td');
		remove_cell_el.classList.add('remove-comment');
		let remove_el = document.createElement('a');
		remove_el.innerText = 'x';
		remove_el.title = 'Remove comment';
		remove_el.addEventListener('click', e => {
			e.preventDefault();
			let xhr = new XMLHttpRequest();
			xhr.comment_id = comment_id;
			xhr.onreadystatechange = () => {
				if (xhr.readyState == XMLHttpRequest.DONE) {
					if (200 == xhr.status) {
						let comment_to_delete = document.querySelector('.comment[data-id="' + xhr.comment_id + '"]');
						if (comment_to_delete) {
							comment_to_delete.remove();
						}
					}
				}
			};
			xhr.open(
				'POST',
				this.comments_uri + '?_' + new Date().getTime(),
				true
			);
			let fd = new FormData();
			fd.append('target_id', this.target_id);
			fd.append('comment_id', comment_id);
			xhr.send(fd);
		});
		remove_cell_el.appendChild(remove_el);
		comment_row_el.appendChild(remove_cell_el);
	}
}
